/************************************************************************
* (c) Copyright Freescale Semiconductor, Inc 2010, All Rights Reserved  *
*************************************************************************

************************************************************************
*                                                                      *
*        Standard Software Flash Driver For FTFL           		         *
*                                                                      *
* FILE NAME     :  SSD_FTFL.h                                          *
* DATE          :  August 16,2010                                      *
*                                                                      *
* AUTHOR        :  FPT Team                                            *
* E-mail        :  b28216@freescale.com                                *
*                                                                      *
************************************************************************/

/************************** CHANGES ***********************************
0.0.1    	06.09.2010    	FPT Team            		Initial Version
0.1.0    	06.11.2010    	FPT Team            		Finalize to 0.1.0
0.1.1     08.16.2010      FPT Team                Update some macros for
                                                  FTFL_P2_256K_256K_4K
                                                  derivative
***********************************************************************/

#ifndef _SSD_FTFL_H_
#define _SSD_FTFL_H_

/* ------------------------ Configuration Macros ------------------------ */

/* For FTFL */
/*	ID	Configuration Name			MCU		P-Flash		P-Flash 	P-Flash 	D-Flash 	D-Flash		    EERAM 	IPS Data	    */		
/*											              Data 	    Size 		  Style 		Data  		Size 	Size            */
/*	1	  FTFL_LL_128K_32K_2K		LLTV2	32-bit		128 KB		P			    32-bit		32 KB		  2 KB	  8-bit		    */		
/*	2	  FTFL_NV_256K_32K_2K		Nevis	64-bit		256 KB		P			    32-bit	 	32 KB 	      2 KB	  32-bit		*/		
/*	3	  FTFL_NV_192K_32K_2K		Nevis	64-bit		192 KB		P			    32-bit		32 KB		  2 KB	  32-bit		*/		
/*	4	  FTFL_NV_128K_32K_2K		Nevis	64-bit		128 KB		P			    32-bit		32 KB		  2 KB  	32-bit		*/		
/*	5	  FTFL_NV_96K_32K_2K		Nevis	64-bit		96 KB		P			    32-bit		32 KB		  2 KB  	32-bit		*/		
/*	6	  FTFL_NV_64K_32K_2K		Nevis	64-bit		64 KB		P			    32-bit		32 KB		  2 KB  	32-bit		*/		
/*	7	  FTFL_P0_64K_32K_2K		P0		64-bit		64 KB		P			    32-bit		32 KB		  2 KB  	32-bit		*/		
/*	8	  FTFL_P0_32K_32K_2K		P0		64-bit		32 KB		P			    32-bit		32 KB		  2 KB  	32-bit		*/		
/*	9	  FTFL_P0_64KS_32K_2K		P0		32-bit		64 KB		S			    32-bit		32 KB		  2 KB  	32-bit		*/		
/*	10	FTFL_P1_256K_32K_2K			P1		64-bit		256 KB		P			    32-bit		32 KB		  2 KB	  32-bit		*/		
/*	11	FTFL_P1_128K_32K_2K			P1		64-bit		128 KB		P			    32-bit		32 KB		  2 KB	  32-bit		*/		
/*	12	FTFL_P1_256KS_32K_2			P1		32-bit		256 KB		S			    32-bit		32 KB		  2 KB	  32-bit		*/		
/*	13	FTFL_P2_256K_256K_4K		P2		64-bit		256 KB		P			    64-bit		256 KB	  	  4 KB	  32-bit		*/		
/*	14	FTFL_P2_128K_256K_4K		P2		64-bit		128 KB		P			    64-bit		256 KB	      4 KB	  32-bit		*/		
/*	15	FTFL_P2_512K_4K				P2		64-bit		512 KB		P									      4 KB	            32-bit		*/		
/*	16	FTFL_P2_384K_4K				P2		64-bit		384 KB		P								  	      4 KB	            32-bit		*/		
/*	17	FTFL_P2_256K_4K				P2		64-bit		256 KB		P									      4 KB	            32-bit		*/		
/*	18	FTFL_P2_128K_4K				P2		64-bit		128 KB		P									      4 KB	            32-bit		*/		
/*	19	FTFL_P2_512KS_4K			P2		64-bit		512 KB		S									      4 KB	            32-bit		*/		
/*	20	FTFL_P2_256KS_4K			P2		64-bit		256 KB		S									      4 KB	            32-bit		*/		
/*	21	FTFL_P3_512K_512K_16K		P3		128-bit		512 KB		P			    128-bit		512 KB	      16 KB 		    32-bit		*/		
/*	22	FTFL_P3_1024K_16K			P3		128-bit		1024 KB		P									      16 KB	            32-bit		*/		
/*	23	FTFL_P3_512K_16K			P3		128-bit		512 KB		P									  	  16 KB	            32-bit		*/		
/*	24	FTFL_P3_384K_16K			P3		128-bit		384 KB		P									  	  16 KB	            32-bit		*/		
/*	25	FTFL_P3_1024KS_16K			P3		128-bit		1024 KB		S									  	  16 KB	            32-bit		*/		
/*	26	FTFL_P3_512KS_16K			P3		128-bit		512 KB		S									      16 KB	            32-bit		*/		

/*Define derivatives*/
#define FTFL_LL_128K_32K_2K			1
#define FTFL_NV_256K_32K_2K			2
#define FTFL_NV_192K_32K_2K			3
#define FTFL_NV_128K_32K_2K			4
#define FTFL_NV_96K_32K_2K			5
#define FTFL_NV_64K_32K_2K			6
#define FTFL_P0_64K_32K_2K			7
#define FTFL_P0_32K_32K_2K			8
#define FTFL_P0_64KS_32K_2K			9
#define FTFL_P1_256K_32K_2K			10
#define FTFL_P1_128K_32K_2K			11
#define FTFL_P1_256KS_32K_2			12
#define FTFL_P2_256K_256K_4K		13
#define FTFL_P2_128K_256K_4K		14
#define FTFL_P2_512K_4K				  15
#define FTFL_P2_384K_4K  		  	16
#define FTFL_P2_256K_4K 			  17
#define FTFL_P2_128K_4K	   		  18
#define FTFL_P2_512KS_4K			  19
#define FTFL_P2_256KS_4K			  20
#define FTFL_P3_512K_512K_16K		21
#define FTFL_P3_1024K_16K			  22
#define FTFL_P3_512K_16K			  23
#define FTFL_P3_384K_16K			  24
#define FTFL_P3_1024KS_16K			25
#define FTFL_P3_512KS_16K			  26

/* Set the derivative */
#define FLASH_DERIVATIVE      FTFL_NV_256K_32K_2K	// for Nevis

#define	BYTE_ADDRESSING

/* Start and End Address of PFlash IFR and DFlash IFR */
	#define PFLASH_IFR_START_ADDRESS		0x000000	// byte address
	#define PFLASH_IFR_END_ADDRESS			0x0000FF
	#define DFLASH_IFR_START_ADDRESS		0x800000
	#define DFLASH_IFR_END_ADDRESS			0x8000FF
	/* The address of the longword that contain EEE Data Set Size and
												   D/E-Flash Partition Code */

	#define DFLASH_IFR_LONGWORD_ADDRESS		0x8000FC	// for EE data size and DE part code



/*------------ Return Code Definition for FTFL SSD -------------------*/

#define FTFL_OK                          0x0000
#define FTFL_ERR_SIZE                    0x0001
#define FTFL_ERR_RANGE                   0x0002
#define FTFL_ERR_ACCERR                  0x0004
#define FTFL_ERR_PVIOL                   0x0008
#define FTFL_ERR_MGSTAT0                 0x0010
#define FTFL_ERR_CHANGEPROT              0x0020
#define FTFL_ERR_EEESIZE                 0x0040
#define FTFL_ERR_EFLASHSIZE              0x0080
#define FTFL_ERR_ADDR                    0x0100
#define FTFL_ERR_NOEEE                   0x0200
#define FTFL_ERR_EFLASHONLY              0x0400
#define FTFL_ERR_DFLASHONLY              0x0800
#define FTFL_ERR_RDCOLERR                0x1000
#define FTFL_ERR_RAMRDY				     0x2000

/* Flash security status */
#define FLASH_SECURITY_STATE_KEYEN         0x80
#define FLASH_SECURITY_STATE_UNSECURED     0x02

#define FLASH_NOT_SECURE                   0x01
#define FLASH_SECURE_BACKDOOR_ENABLED      0x02
#define FLASH_SECURE_BACKDOOR_DISABLED     0x04

/* EERAM Function Control Code */
#define EEE_ENABLE						             0x00
#define EEE_DISABLE					             	 0xFF

/* Total no. of backdoor key */
#define TOTAL_KEY_BYTES                    0x08

/* Maximum value for program & read once phrase index */
#define MAX_PHRASE_INDEX                   0x0F

/*-------------- Read/Write/Set/Clear Operation Macros ------------------*/

#define REG_BIT_SET(address, mask)      (*(VUINT8*)(address) |= (mask))
#define REG_BIT_CLEAR(address, mask)    (*(VUINT8*)(address) &= ~(mask))
#define REG_BIT_TEST(address, mask)     (*(VUINT8*)(address) & (mask))

#define REG_BIT_SET16(address, mask)      (*(VUINT16*)(address) |= (mask))
#define REG_BIT_CLEAR16(address, mask)    (*(VUINT16*)(address) &= ~(mask))
#define REG_BIT_TEST16(address, mask)     (*(VUINT16*)(address) & (mask))

#define REG_WRITE(address, value)       (*(VUINT8*)(address) = (value))
#define REG_WRITE16(address, value)      (*(VUINT16*)(address) = (value))
#define REG_READ(address)               ((UINT8)(*(VUINT8*)(address)))
#define REG_READ16(address)             ((UINT16)(*(VUINT8*)(address)))
#define REG_WRITE32(address, value)     (*(VUINT32*)(address) = (value))
#define REG_READ32(address)             ((UINT32)(*(VUINT32*)(address)))

#define WRITE8(address, value)          (*(VUINT8*)(address) = (value))
#define READ8(address)                  ((UINT8)(*(VUINT8*)(address)))
#define SET8(address, value)            (*(VUINT8*)(address) |= (value))
#define CLEAR8(address, value)          (*(VUINT8*)(address) &= ~(value))
#define TEST8(address, value)           (*(VUINT8*)(address) & (value))

#define WRITE16(address, value)         (*(VUINT16*)(address) = (value))
#define READ16(address)                 ((UINT16)(*(VUINT16*)(address)))
#define SET16(address, value)           (*(VUINT16*)(address) |= (value))
#define CLEAR16(address, value)         (*(VUINT16*)(address) &= ~(value))
#define TEST16(address, value)          (*(VUINT16*)(address) & (value))

#define WRITE32(address, value)         (*(VUINT32*)(address) = (value))
#define READ32(address)                 ((UINT32)(*(VUINT8*)(address)))
//#define READ32(address)                 ((UINT32)(*(VUINT32*)(address)))
#define SET32(address, value)           (*(VUINT32*)(address) |= (value))
#define CLEAR32(address, value)         (*(VUINT32*)(address) &= ~(value))
#define TEST32(address, value)          (*(VUINT32*)(address) & (value))

#ifdef NEVIS2 // Nevis
	#define bits_31_24(data)                ((data & (0x00FFL<<24)) >> 24)
	#define bits_23_16(data)                ((data & (0x00FFL<<16)) >> 16)
	#define bits_15_8(data)                 ((data & (0x00FFL<<8)) >> 8)
	#define bits_7_0(data)                  (data & 0xFF)
#else
	#define bits_31_24(data)                ((data & (UINT32)(0xFF<<24)) >> 24)
	#define bits_23_16(data)                ((data & (UINT32)(0xFF<<16)) >> 16)
	#define bits_15_8(data)                 ((data & (UINT32)(0xFF<<8)) >> 8)
	#define bits_7_0(data)                  (UINT32)(data & 0xFF)
#endif

/* Word size */
#define FTFL_WORD_SIZE                 0x0002     /* 2 bytes */
  
/* Longword size */
#define FTFL_LONGWORD_SIZE             0x0004     /* 4 bytes */

/* Phrase size */
#define FTFL_PHRASE_SIZE               0x0008     /* 8 bytes */

/*--------------------- CallBack function period ------------------------*/
#define FLASH_CALLBACK_CS                1          /* Check Sum */

/*-------------------- Callback function prototype ---------------------*/
typedef void (* PCALLBACK)(void);

/*--------------------Null Callback function defination ------------------*/
#define NULL_CALLBACK                    ((PCALLBACK)0xFFFFFFFF)

/*---------------- Flash SSD Configuration Structure -------------------*/
typedef struct _ssd_config
{
    UINT32 		ftflRegBase;                     /* FTFL control register base */
    UINT32 		PFlashBlockBase;                 /* base address of PFlash block */
    UINT32 		PFlashBlockSize;                 /* size of PFlash block */
    UINT32 		DFlashBlockBase;                 /* base address of DFlash block */
    UINT32 		DFlashBlockSize;                 /* size of DFlash block */
    UINT32 		EERAMBlockBase;                  /* base address of EERAM block */
    UINT32 		EERAMBlockSize;                  /* size of EERAM block */
    UINT32 		EEEBlockSize;                    /* size of EEE block */
    BOOL   		DebugEnable;                     /* background debug mode enable bit */
    PCALLBACK   CallBack;                      /* pointer to callback function */
} FLASH_SSD_CONFIG, *PFLASH_SSD_CONFIG;


/* Flash block size */
#if(FTFL_LL_128K_32K_2K == FLASH_DERIVATIVE)    /* (LLTV2) */
  /* FTFL module base */
  #define FTFL_REG_BASE       0xFFFF82E0
  #define DFLASH_IFR_BASE				0x00800000	
  #define PFLASH_BLOCK_BASE			0x00000000
  #define DEFLASH_BLOCK_BASE	  0x00200000
  #define EERAM_BLOCK_BASE			0x00280000

	/* Flash sector size */
  #define FTFL_SECTOR_SIZE    0x00000400      /* 1 KB size   */
	#define PBLOCK_SIZE           0x00020000      /* 128 KB size */
	#define DEBLOCK_SIZE          0x00008000      /* 32 KB size  */
	#define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */
	/* EEE Data Set Size Field Description */     
  #define EEESIZE_0000				0x00000000      /* Reserved   */
  #define EEESIZE_0001			  	0x00000000      /* Reserved   */
  #define EEESIZE_0010			  	0x00000000      /* Reserved   */
  #define EEESIZE_0011			  	0x00000800
  #define EEESIZE_0100		  		0x00000400
  #define EEESIZE_0101			  	0x00000200
  #define EEESIZE_0110			  	0x00000100
  #define EEESIZE_0111			  	0x00000080
  #define EEESIZE_1000			  	0x00000040
  #define EEESIZE_1001			  	0x00000020
  #define EEESIZE_1010				0x00000000      /* Reserved   */
  #define EEESIZE_1011			  	0x00000000      /* Reserved   */
  #define EEESIZE_1100			  	0x00000000      /* Reserved   */
  #define EEESIZE_1101			  	0x00000000      /* Reserved   */
  #define EEESIZE_1110			  	0x00000000      /* Reserved   */
  #define EEESIZE_1111			  	0x00000000      /* Default value */

  /* D/E-Flash Partition Code Field Description */
  #define DEPART_0000				  	0x00008000
  #define DEPART_0001				  	0x00006000
  #define DEPART_0010				  	0x00004000
  #define DEPART_0011				  	0x00000000
  #define DEPART_0100				  	0x00008000      /* Reserved   */
  #define DEPART_0101				  	0x00008000      /* Reserved   */
  #define DEPART_0110				  	0x00008000      /* Reserved   */
  #define DEPART_0111				  	0x00008000      /* Reserved   */
  #define DEPART_1000				  	0x00000000      
  #define DEPART_1001				  	0x00002000      
  #define DEPART_1010				  	0x00004000      
  #define DEPART_1011				  	0x00008000      
  #define DEPART_1100				  	0x00008000      /* Reserved   */
  #define DEPART_1101				  	0x00008000      /* Reserved   */
  #define DEPART_1110				  	0x00008000      /* Reserved   */
  #define DEPART_1111				  	0x00008000      /* Default value */
      
#elif(FTFL_NV_256K_32K_2K == FLASH_DERIVATIVE)  /* (Nevis)     */
/* For Nevis only */
  /* FTFL module base */
#ifdef	BYTE_ADDRESSING
  #define FTFL_REG_BASE       	  	0x0001C780	 /* byte address of FTFL registers */
  
  /* Flash block base address for program memory space */
  #define DFLASH_IFR_BASE			0x00080000	 /* ??? */
  #define PFLASH_BLOCK_BASE	   		0x00000000	 /* byte address for 1st bank of p flash */
  #define DEFLASH_BLOCK_BASE	  	0x000D0000 	 /* byte address for 2nd bank of d flash/FlexNVM */
  #define EERAM_BLOCK_BASE			0x0003C000
  
  /* Flash sector size for FlexNVM/data flash */
  #define FTFL_DFLASH_SECTOR_SIZE      0x00000400      /* 1 KB size   */
  /* Flash sector size for program flash*/
  #define FTFL_SECTOR_SIZE      	  0x00000800      /* 2 KB size   */
  

  /* Flash block size */
  #define PBLOCK_SIZE           0x00040000      /* 256 KB size for p flash*/
  #define DEBLOCK_SIZE          0x00008000      /* 32 KB size for d flash */
  #define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */

  /* Flash block base address for data memory space*/  
  #define PFLASH2_BLOCK_BASE	   	0x00040000	 /* byte address for 1nd bank of program flash */
  #define DEFLASH2_BLOCK_BASE	  	0x00010000	 /* byte address for 2nd bank of data flash */
  #define PBLOCK2_SIZE           	0x00040000      /* 256 KB size for p flash*/
  #define DEBLOCK2_SIZE          	0x00008000      /* 32 KB size for d flash */  
  
#else
  #define FTFL_REG_BASE       	  	0x0000E3C0	 /* word address */
  #define DFLASH_IFR_BASE			0x00040000	 /* ??? */
  #define PFLASH_BLOCK_BASE	   		0x00000000
  #define DEFLASH_BLOCK_BASE	  	0x00020000	
  #define EERAM_BLOCK_BASE			0x0001E000

  
  /* Flash sector size */
  #define FTFL_SECTOR_SIZE      0x00000400      /* 2 KB size   */
  #define PBLOCK_SIZE           0x00040000      /* 256 KB size */
  #define DEBLOCK_SIZE          0x00004000      /* 32 KB size  */
  #define EERAM_BLOCK_SIZE      0x00000400      /* 2 KB size   */
#endif  

  /* EEE Data Set Size Field Description */
  #define EEESIZE_0000				0x00000000		/* reserved */      
  #define EEESIZE_0001			  	0x00000000		/* reserved */        
  #define EEESIZE_0010			  	0x00000000 		/* reserved */       
  #define EEESIZE_0011			  	0x00000800      
  #define EEESIZE_0100		  		0x00000400      
  #define EEESIZE_0101			  	0x00000200      
  #define EEESIZE_0110			  	0x00000100      
  #define EEESIZE_0111			  	0x00000080      
  #define EEESIZE_1000			  	0x00000040      
  #define EEESIZE_1001			  	0x00000020      
  #define EEESIZE_1010				0x00000000      /* Reserved    */
  #define EEESIZE_1011			  	0x00000000      /* Reserved    */
  #define EEESIZE_1100			  	0x00000000      /* Reserved    */
  #define EEESIZE_1101			  	0x00000000      /* Reserved    */
  #define EEESIZE_1110			  	0x00000000      /* Reserved    */
  #define EEESIZE_1111			  	0x00000000      /* Default value */
  
  /* D/E-Flash Partition Code Field Description */
  #define DEPART_0000				    0x00008000		/* 32KB data flash, no EEPROM */
  #define DEPART_0001				  	0x00006000      /* 24KB data flash, 8K EEE */
  #define DEPART_0010				  	0x00004000		/* 16KB data flash, 16K EEE   */
  #define DEPART_0011				  	0x00000000		/* 0 data flash,32K EEE    */
  #define DEPART_0100				  	0x00000000  	/* Reserved    */
  #define DEPART_0101				  	0x00000000  	/* Reserved    */
  #define DEPART_0110				  	0x00000000  	/* Reserved    */
  #define DEPART_0111				  	0x00000000      /* Reserved    */
  #define DEPART_1000				  	0x00000000		/* 0 data flash,32K EEE    */
  #define DEPART_1001				  	0x00002000      
  #define DEPART_1010				  	0x00004000
  #define DEPART_1011				  	0x00008000
  #define DEPART_1100				  	0x00000000  	/* Reserved    */
  #define DEPART_1101				  	0x00000000  	/* Reserved    */
  #define DEPART_1110				  	0x00000000  	/* Reserved    */
  #define DEPART_1111				  	0x00000000      /* Reserved    */

	
#elif(FTFL_NV_192K_32K_2K == FLASH_DERIVATIVE)  /* (Nevis)     */
	#define PBLOCK_SIZE           0x00030000      /* 192 KB size */
	#define DEBLOCK_SIZE          0x00008000      /* 32 KB size  */
	#define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */
	
#elif(FTFL_NV_128K_32K_2K == FLASH_DERIVATIVE)  /* (Nevis)     */
 	#define PBLOCK_SIZE           0x00020000      /* 128 KB size */
	#define DEBLOCK_SIZE         	0x00008000      /* 32 KB size  */
	#define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */
	
#elif(FTFL_NV_96K_32K_2K == FLASH_DERIVATIVE)   /* (Nevis)     */
 	#define PBLOCK_SIZE           0x00018000      /* 96 KB size  */
	#define DEBLOCK_SIZE         	0x00008000      /* 32 KB size  */
	#define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */
	
#elif(FTFL_NV_64K_32K_2K == FLASH_DERIVATIVE)   /* (Nevis)     */
  #define PBLOCK_SIZE          	0x00010000      /* 64 KB size  */
	#define DEBLOCK_SIZE          0x00008000      /* 32 KB size  */
	#define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */
	
#elif(FTFL_P0_64K_32K_2K == FLASH_DERIVATIVE)   /* (P0)        */
  #define PBLOCK_SIZE           0x00010000      /* 64 KB size  */
	#define DEBLOCK_SIZE          0x00008000      /* 32 KB size  */
	#define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */
  
#elif(FTFL_P0_32K_32K_2K == FLASH_DERIVATIVE)   /* (P0)        */
  #define PBLOCK_SIZE           0x00008000      /* 32 KB size  */
	#define DEBLOCK_SIZE          0x00008000      /* 32 KB size  */
	#define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */
  
#elif(FTFL_P0_64KS_32K_2K == FLASH_DERIVATIVE)  /* (P0)        */
  #define PBLOCK_SIZE           0x00010000      /* 64 KB size  */
	#define DEBLOCK_SIZE          0x00008000      /* 32 KB size  */
	#define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */
  
#elif(FTFL_P1_256K_32K_2K == FLASH_DERIVATIVE)  /* (P1)        */
  #define PBLOCK_SIZE           0x00040000      /* 256 KB size */
	#define DEBLOCK_SIZE          0x00008000      /* 32 KB size  */
	#define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */
  
#elif(FTFL_P1_128K_32K_2K == FLASH_DERIVATIVE)  /* (P1)        */
  #define PBLOCK_SIZE           0x00020000      /* 128 KB size */
	#define DEBLOCK_SIZE          0x00008000      /* 32 KB size  */
	#define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */
  
#elif(FTFL_P1_256KS_32K_2 == FLASH_DERIVATIVE)  /* (P1)        */
  #define PBLOCK_SIZE           0x00040000      /* 256 KB size */
  #define DEBLOCK_SIZE          0x00008000      /* 32 KB size  */
  #define EERAM_BLOCK_SIZE      0x00000800      /* 2 KB size   */
  
#elif(FTFL_P2_256K_256K_4K == FLASH_DERIVATIVE) /* (P2)        */
  /* FTFL module base */
  #define FTFL_REG_BASE       0x40020000
  #define DFLASH_IFR_BASE	  0x00800000	
  #define PFLASH_BLOCK_BASE	   0x00000000
  #define DEFLASH_BLOCK_BASE	  0x10000000
  #define EERAM_BLOCK_BASE			0x14000000
  
  /* Flash sector size */
  #define FTFL_SECTOR_SIZE    0x00000800      /* 2 KB size   */
  #define PBLOCK_SIZE           0x00040000      /* 256 KB size */
  #define DEBLOCK_SIZE          0x00040000      /* 256 KB size */
  #define EERAM_BLOCK_SIZE      0x00001000      /* 4 KB size   */
  
  /* EEE Data Set Size Field Description */
  #define EEESIZE_0000				0x00000000      /* Reserved    */
  #define EEESIZE_0001			  	0x00000000      /* Reserved    */
  #define EEESIZE_0010			  	0x00001000      
  #define EEESIZE_0011			  	0x00000800      
  #define EEESIZE_0100		  		0x00000400      
  #define EEESIZE_0101			  	0x00000200      
  #define EEESIZE_0110			  	0x00000100      
  #define EEESIZE_0111			  	0x00000080      
  #define EEESIZE_1000			  	0x00000040      
  #define EEESIZE_1001			  	0x00000020      
  #define EEESIZE_1010				  0x00000000      /* Reserved    */
  #define EEESIZE_1011			  	0x00000000      /* Reserved    */
  #define EEESIZE_1100			  	0x00000000      /* Reserved    */
  #define EEESIZE_1101			  	0x00000000      /* Reserved    */
  #define EEESIZE_1110			  	0x00000000      /* Reserved    */
  #define EEESIZE_1111			  	0x00000000      /* Default value */
  
  /* D/E-Flash Partition Code Field Description */
  #define DEPART_0000				  	0x00040000
  #define DEPART_0001				  	0x00040000      /* Reserved    */
  #define DEPART_0010				  	0x0003C000
  #define DEPART_0011				  	0x00038000
  #define DEPART_0100				  	0x00030000  
  #define DEPART_0101				  	0x00020000  
  #define DEPART_0110				  	0x00000000  
  #define DEPART_0111				  	0x00040000      /* Reserved    */
  #define DEPART_1000				  	0x00000000
  #define DEPART_1001				  	0x00040000      /* Reserved    */
  #define DEPART_1010				  	0x00004000
  #define DEPART_1011				  	0x00008000
  #define DEPART_1100				  	0x00010000  
  #define DEPART_1101				  	0x00020000  
  #define DEPART_1110				  	0x00040000  
  #define DEPART_1111				  	0x00040000      /* Reserved    */
        
#elif(FTFL_P2_128K_256K_4K == FLASH_DERIVATIVE) /* (P2)        */
  #define PBLOCK_SIZE           0x00020000      /* 128 KB size */
	#define DEBLOCK_SIZE          0x00040000      /* 256 KB size */
	#define EERAM_BLOCK_SIZE      0x00001000      /* 4 KB size   */
  
#elif(FTFL_P2_512K_4K == FLASH_DERIVATIVE)      /* (P2)        */
  #define PBLOCK_SIZE           0x00080000      /* 512 KB size */
  #define DEBLOCK_SIZE          0x00000000      /* 0 KB size   */
  #define EERAM_BLOCK_SIZE      0x00001000      /* 4 KB size   */
  
#elif(FTFL_P2_384K_4K == FLASH_DERIVATIVE)      /* (P2)        */
  #define PBLOCK_SIZE           0x00060000      /* 384 KB size */
  #define DEBLOCK_SIZE          0x00000000      /* 0 KB size   */
	#define EERAM_BLOCK_SIZE      0x00001000      /* 4 KB size   */
  
#elif(FTFL_P2_256K_4K == FLASH_DERIVATIVE)      /* (P2)        */
  #define PBLOCK_SIZE           0x00040000      /* 256 KB size */
	#define DEBLOCK_SIZE          0x00000000      /* 0 KB size   */
	#define EERAM_BLOCK_SIZE      0x00001000      /* 4 KB size   */
  
#elif(FTFL_P2_128K_4K == FLASH_DERIVATIVE)      /* (P2)        */
  #define PBLOCK_SIZE           0x00020000      /* 128 KB size */
  #define DEBLOCK_SIZE          0x00000000      /* 0 KB size   */
  #define EERAM_BLOCK_SIZE      0x00001000      /* 4 KB size   */
  
#elif(FTFL_P2_512KS_4K == FLASH_DERIVATIVE)     /* (P2)        */
  #define PBLOCK_SIZE           0x00080000      /* 512 KB size */
	#define DEBLOCK_SIZE          0x00000000      /* 0 KB size   */
  #define EERAM_BLOCK_SIZE      0x00001000      /* 4 KB size   */
  
#elif(FTFL_P2_256KS_4K == FLASH_DERIVATIVE)     /* (P2)        */
  #define PBLOCK_SIZE           0x00040000      /* 256 KB size */
  #define DEBLOCK_SIZE          0x00000000      /* 0 KB size   */
  #define EERAM_BLOCK_SIZE      0x00001000      /* 4 KB size   */
  
#elif(FTFL_P3_512K_512K_16K == FLASH_DERIVATIVE) /* (P3)       */
  #define PBLOCK_SIZE           0x00080000      /* 512 KB size */
	#define DEBLOCK_SIZE          0x00080000      /* 512 KB size */
	#define EERAM_BLOCK_SIZE      0x00004000      /* 16 KB size  */
  
#elif(FTFL_P3_1024K_16K == FLASH_DERIVATIVE)    /* (P3)         */
  #define PBLOCK_SIZE           0x00100000      /* 1024 KB size */
  #define DEBLOCK_SIZE          0x00000000      /* 0 KB size    */
  #define EERAM_BLOCK_SIZE      0x00004000      /* 16 KB size   */
  
#elif(FTFL_P3_512K_16K == FLASH_DERIVATIVE)     /* (P3)         */
  #define PBLOCK_SIZE          	0x00080000      /* 512 KB size  */
	#define DEBLOCK_SIZE          0x00000000      /* 0 KB size    */
  #define EERAM_BLOCK_SIZE      0x00004000      /* 16 KB size   */
  
#elif(FTFL_P3_384K_16K == FLASH_DERIVATIVE)     /* (P3)         */
  #define PBLOCK_SIZE           0x00060000      /* 384 KB size  */
  #define DEBLOCK_SIZE          0x00000000      /* 0 KB size    */
	#define EERAM_BLOCK_SIZE      0x00004000      /* 16 KB size   */
  
#elif(FTFL_P3_1024KS_16K == FLASH_DERIVATIVE)   /* (P3)         */
 	#define PBLOCK_SIZE           0x00100000      /* 1024 KB size */
	#define DEBLOCK_SIZE          0x00000000      /* 0 KB size    */
	#define EERAM_BLOCK_SIZE      0x00004000      /* 16 KB size   */
  
#elif(FTFL_P3_512KS_16K == FLASH_DERIVATIVE)    /* (P3)         */
  #define PBLOCK_SIZE           0x00080000      /* 512 KB size  */
	#define DEBLOCK_SIZE          0x00000000      /* 0 KB size    */
	#define EERAM_BLOCK_SIZE      0x00004000      /* 16 KB size   */
#endif

/* -------------------- Function Pointer ------------------------------- */
typedef UINT32 (*pFLASHCOMMANDSEQUENCE) (PFLASH_SSD_CONFIG PSSDConfig, \
                                         UINT8 index, \
                                         UINT8* pCommandArray);

/* Flash initialization */
typedef UINT32 (*pFLASHINIT) (PFLASH_SSD_CONFIG PSSDConfig);

/* P-Flash get protection */
typedef UINT32 (*pPFLASHGETPROTECTION) (PFLASH_SSD_CONFIG PSSDConfig, \
													              UINT32*  protectStatus);
                                              
/* P-Flash set protection */
typedef UINT32 (*pPFLASHSETPROTECTION) (PFLASH_SSD_CONFIG  PSSDConfig, \
                                        UINT32  protectStatus);
														
/* Flash get interrupt enable */
typedef UINT32 (*pFLASHGETINTERRUPTENABLE) (PFLASH_SSD_CONFIG PSSDConfig, \
                                            UINT8* interruptState);

/* Flash set interrupt enable */
typedef UINT32 (*pFLASHSETINTERRUPTENABLE) (PFLASH_SSD_CONFIG PSSDConfig, \
                                            UINT8 interruptState);

/* Flash get security state */
typedef UINT32 (*pFLASHGETSECURITYSTATE) (PFLASH_SSD_CONFIG PSSDConfig, \
                                          UINT8* securityState);
														
/* Flash security bypass */
typedef UINT32 (*pFLASHSECURITYBYPASS) (PFLASH_SSD_CONFIG PSSDConfig, \
                                        UINT8* keyBuffer, \
                                        pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* Flash erase all Block */
typedef UINT32 (*pFLASHERASEALLBLOCK) (PFLASH_SSD_CONFIG PSSDConfig, \
														           pFLASHCOMMANDSEQUENCE FlashCommandSequence);
								
/* Flash erase Block */
typedef UINT32 (*pFLASHERASEBLOCK) (PFLASH_SSD_CONFIG PSSDConfig, \
														        UINT32 destination, \
														        pFLASHCOMMANDSEQUENCE FlashCommandSequence);
									
/* Flash erase sector */
typedef UINT32 (*pFLASHERASESECTOR) (PFLASH_SSD_CONFIG PSSDConfig, \
														         UINT32 destination, \
														         UINT32 size, \
														         pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* Flash erase suspend */
typedef UINT32 (*pFLASHERASESUSPEND) (PFLASH_SSD_CONFIG PSSDConfig);

/* Flash erase resume */
typedef UINT32 (*pFLASHERASERESUME) (PFLASH_SSD_CONFIG PSSDConfig);													   
	
/* Flash program Section */
typedef UINT32 (*pFLASHPROGRAMSECTION) (PFLASH_SSD_CONFIG PSSDConfig, \
														            UINT32 destination, \
														            UINT16 Number, \
														            pFLASHCOMMANDSEQUENCE FlashCommandSequence);												
												
/* Flash program Longword */
typedef UINT32 (*pFLASHPROGRAMLONGWORD) (PFLASH_SSD_CONFIG PSSDConfig, \
														             UINT32 destination, \
													             	 UINT32 size, \
														             UINT32 source, \
														             pFLASHCOMMANDSEQUENCE FlashCommandSequence);
										   
/* Flash program Word */
typedef UINT32 (*pFLASHPROGRAMWORD) (PFLASH_SSD_CONFIG PSSDConfig, \
														         UINT32 destination, \
														         UINT32 size, \
														         UINT32 source, \
														         pFLASHCOMMANDSEQUENCE FlashCommandSequence);										   
												
/* Flash program Byte */
typedef UINT32 (*pFLASHPROGRAMBYTE) (PFLASH_SSD_CONFIG PSSDConfig, \
														         UINT32 destination, \
														         UINT32 size, \
														         UINT32 source, \
														         pFLASHCOMMANDSEQUENCE FlashCommandSequence);	
												
/* Flash checksum */
typedef UINT32 (*pFLASHCHECKSUM) (PFLASH_SSD_CONFIG PSSDConfig, \
													      	UINT32 destination, \
														      UINT32 size, \
													 	      UINT32* PSum);
										   
/* Flash verify all block */
typedef UINT32 (*pFLASHVERIFYALLBLOCK) (PFLASH_SSD_CONFIG PSSDConfig, \
														            UINT8 marginLevel, \
														            pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* Flash verify block */
typedef UINT32 (*pFLASHVERIFYBLOCK) (PFLASH_SSD_CONFIG PSSDConfig, \
													        	UINT32 destination, \
													        	UINT8 marginLevel, \
													        	pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* Flash verify section*/
typedef UINT32 (*pFLASHVERIFYSECTION) (PFLASH_SSD_CONFIG PSSDConfig, \
													          	UINT32 destination, \
														          UINT16 Number, \
													          	UINT8 marginLevel, \
														          pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* Flash read once */
typedef UINT32 (*pFLASHREADONCE) (PFLASH_SSD_CONFIG PSSDConfig, \
														      UINT8* pDataArray, \
														      pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* Flash program once */
typedef UINT32 (*pFLASHPROGRAMONCE) (PFLASH_SSD_CONFIG PSSDConfig, \
													         	UINT8* pDataArray, \
													        	pFLASHCOMMANDSEQUENCE FlashCommandSequence);
/* Flash program check */
typedef UINT32 (*pFLASHPROGRAMCHECK) (PFLASH_SSD_CONFIG PSSDConfig, \
              												UINT32  destination, \
              												UINT32  size, \
              												UINT8*  pExpectedData, \
              												UINT32* pFailAddr, \
              												UINT8*  pFailData, \
              												UINT8   marginLevel, \
              												pFLASHCOMMANDSEQUENCE FlashCommandSequence);
              										
/* Flash read resource */
typedef UINT32 (*pFLASHREADRESOURCE) (PFLASH_SSD_CONFIG PSSDConfig, \
          														UINT32 destination, \
          														UINT8* pDataArray, \
          														pFLASHCOMMANDSEQUENCE FlashCommandSequence);

typedef int (*ptest) (int aa, int bb);
          																					


/*---------------- Function Prototypes for Flash SSD --------------------*/

/* Flash initialization */
extern UINT32 FlashInit(PFLASH_SSD_CONFIG PSSDConfig);

/* Internal function. Called by driver APIs only */
/* Flash command sequence */

extern UINT32 FlashCommandSequence(PFLASH_SSD_CONFIG PSSDConfig, \
                                  UINT8 index, \
                                  UINT8* pCommandArray);

/* P-Flash get protection */
extern UINT32 PFlashGetProtection(PFLASH_SSD_CONFIG PSSDConfig, \
                                  UINT32*  protectStatus);      

/* P-Flash set protection */
extern UINT32 PFlashSetProtection(PFLASH_SSD_CONFIG PSSDConfig, \
                                  UINT32  protectStatus);												
                                                
/* Flash get interrupt enable */
extern UINT32 FlashGetInterruptEnable(PFLASH_SSD_CONFIG PSSDConfig, \
                                      UINT8* interruptState);

/* Flash set interrupt enable */
extern UINT32 FlashSetInterruptEnable(PFLASH_SSD_CONFIG PSSDConfig, \
                                      UINT8 interruptState);

/* Flash get security state */
extern UINT32 FlashGetSecurityState(PFLASH_SSD_CONFIG PSSDConfig, \
                                    UINT8* securityState);

/* Flash security bypass */
extern UINT32 FlashSecurityBypass(PFLASH_SSD_CONFIG PSSDConfig, \
                                  UINT8* keyBuffer, \
                                  pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* Flash erase all Block */
extern UINT32 FlashEraseAllBlock(PFLASH_SSD_CONFIG PSSDConfig, \
												         pFLASHCOMMANDSEQUENCE FlashCommandSequence);
								
/* Flash erase Block */
extern UINT32 FlashEraseBlock(PFLASH_SSD_CONFIG PSSDConfig, \
      												UINT32 destination, \
      												pFLASHCOMMANDSEQUENCE FlashCommandSequence);
									
/* Flash erase sector */
extern UINT32 FlashEraseSector(PFLASH_SSD_CONFIG PSSDConfig, \
        											UINT32 destination, \
        											UINT32 size, \
        											pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* Flash erase suspend */
extern UINT32 FlashEraseSuspend(PFLASH_SSD_CONFIG PSSDConfig);

/* Flash erase resume */
extern UINT32 FlashEraseResume(PFLASH_SSD_CONFIG PSSDConfig);
												
/* Flash program Section */
extern UINT32 FlashProgramSection(PFLASH_SSD_CONFIG PSSDConfig, \
          												UINT32 destination, \
          												UINT16 Number, \
          												pFLASHCOMMANDSEQUENCE FlashCommandSequence);												
												
/* Flash program Longword */
extern UINT32 FlashProgramLongword(PFLASH_SSD_CONFIG PSSDConfig, \
            												UINT32 destination, \
            												UINT32 size, \
            												UINT32 source, \
            												pFLASHCOMMANDSEQUENCE FlashCommandSequence);
										   
/* Flash program Word */
extern UINT32 FlashProgramWord(PFLASH_SSD_CONFIG PSSDConfig, \
        												UINT32 destination, \
        												UINT32 size, \
        												UINT32 source, \
        												pFLASHCOMMANDSEQUENCE FlashCommandSequence);										   
        				 							 
/* Flash program Byte */
extern UINT32 FlashProgramByte(PFLASH_SSD_CONFIG PSSDConfig, \
        												UINT32 destination, \
        												UINT32 size, \
        												UINT32 source, \
        												pFLASHCOMMANDSEQUENCE FlashCommandSequence);										 
	
/* Flash checksum */
extern UINT32 FlashCheckSum(PFLASH_SSD_CONFIG PSSDConfig, \
                           UINT32 destination, \
                           UINT32 size, \
                           UINT32* PSum);
										   
/* Flash verify all block */
extern UINT32 FlashVerifyAllBlock(PFLASH_SSD_CONFIG PSSDConfig, \
          												UINT8 marginLevel, \
          												pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* Flash verify block */
extern UINT32 FlashVerifyBlock(PFLASH_SSD_CONFIG PSSDConfig, \
        												UINT32 destination, \
        												UINT8 marginLevel, \
        												pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* Flash verify section*/
extern UINT32 FlashVerifySection(PFLASH_SSD_CONFIG PSSDConfig, \
          												UINT32 destination, \
          												UINT16 Number, \
          												UINT8 marginLevel, \
          												pFLASHCOMMANDSEQUENCE FlashCommandSequence);		

/* Flash read once */
extern UINT32 FlashReadOnce(PFLASH_SSD_CONFIG PSSDConfig, \
      											UINT8* pDataArray, \
      											pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* Flash program once */
extern UINT32 FlashProgramOnce(PFLASH_SSD_CONFIG PSSDConfig, \
        											UINT8* pDataArray, \
        											pFLASHCOMMANDSEQUENCE FlashCommandSequence);
/* Flash program check */
extern UINT32 FlashProgramCheck(PFLASH_SSD_CONFIG PSSDConfig, \
        												UINT32  destination, \
        												UINT32  size, \
        												UINT8*  pExpectedData, \
        												UINT32* pFailAddr, \
        												UINT8*  pFailData, \
        												UINT8   marginLevel, \
        												pFLASHCOMMANDSEQUENCE FlashCommandSequence);
										
/* Flash read resource */
extern UINT32 FlashReadResource(PFLASH_SSD_CONFIG PSSDConfig, \
        												UINT32 destination, \
        												UINT8* pDataArray, \
        												pFLASHCOMMANDSEQUENCE FlashCommandSequence);	

extern int test (int aa, int bb);

#if (DEBLOCK_SIZE)
/* D-Flash get protection */
typedef UINT32 (*pDFLASHGETPROTECTION) (PFLASH_SSD_CONFIG PSSDConfig, \
											                  UINT8*  protectStatus);    
/* D-Flash set protection */
typedef UINT32 (*pDFLASHSETPROTECTION) (PFLASH_SSD_CONFIG PSSDConfig, \
										                  	UINT8  protectStatus);
									
/* EERAM get protection */
 typedef UINT32 (*pEERAMGETPROTECTION) (PFLASH_SSD_CONFIG PSSDConfig, \
											                  UINT8*  protectStatus);      

/* EERAM set protection */
typedef UINT32 (*pEERAMSETPROTECTION) (PFLASH_SSD_CONFIG PSSDConfig, \
											                 UINT8  protectStatus);
			/* Flash D/E-Flash Partition */
typedef UINT32 (*pDEFLASHPARTITION) (PFLASH_SSD_CONFIG PSSDConfig, \
														         UINT8 EEEDataSizeCode, \
														         UINT8 DEPartitionCode, \
													         	 pFLASHCOMMANDSEQUENCE FlashCommandSequence);
/* Flash Set EEEEnable */
typedef UINT32 (*pSETEEEENABLE) (PFLASH_SSD_CONFIG PSSDConfig, \
														     UINT8 EEEEnable);
												
/* Get EEPROM Emulator Configuration*/
typedef UINT32 (*pGETEEECONFIG) (PFLASH_SSD_CONFIG PSSDConfig, \
														     UINT8* EEEDataSize, \
														     UINT8* DEPartitionCode, \
														     pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* EEPROM Emulator Write */
typedef UINT32 (*pEEEWRITE) (PFLASH_SSD_CONFIG PSSDConfig, \
													 	 UINT32 destination, \
														 UINT32 size, \
														 UINT32 source);	  
/* D-Flash get protection */
extern UINT32 DFlashGetProtection(PFLASH_SSD_CONFIG PSSDConfig, \
                                  UINT8*  protectStatus);      

/* D-Flash set protection */
extern UINT32 DFlashSetProtection(PFLASH_SSD_CONFIG PSSDConfig, \
                                  UINT8  protectStatus);
												
/* EERAM get protection */
extern UINT32 EERAMGetProtection(PFLASH_SSD_CONFIG PSSDConfig, \
                                 UINT8*  protectStatus);      

/* EERAM set protection */
extern UINT32 EERAMSetProtection(PFLASH_SSD_CONFIG PSSDConfig, \
                                 UINT8  protectStatus);	
/* Flash D/E-Flash Partition */
extern UINT32 DEFlashPartition(PFLASH_SSD_CONFIG PSSDConfig, \
												       UINT8 EEEDataSizeCode, \
												       UINT8 DEPartitionCode, \
												       pFLASHCOMMANDSEQUENCE FlashCommandSequence);
/* Flash Set EEEEnable */
extern UINT32 SetEEEEnable(PFLASH_SSD_CONFIG PSSDConfig, \
												   UINT8 EEEEnable);
												
/* Get EEPROM Emulator Configuration*/
extern UINT32 GetEEEConfig(PFLASH_SSD_CONFIG PSSDConfig, \
												   UINT8* EEEDataSize, \
												   UINT8* DEPartitionCode, \
												   pFLASHCOMMANDSEQUENCE FlashCommandSequence);

/* EEPROM Emulator Write */
extern UINT32 EEEWrite(PFLASH_SSD_CONFIG PSSDConfig, \
											 UINT32 destination, \
											 UINT32 size, \
											 UINT32 source);
#endif //of DEBLOCK_SIZE											   

#endif /* _SSD_FTFL_H_ */


